<?php

namespace App\Http\Controllers\Api;

use App\Enums\KYCStatus;
use App\Enums\TxnStatus;
use App\Enums\TxnType;
use App\Http\Controllers\Controller;
use App\Http\Resources\NavigationResource;
use App\Http\Resources\NotificationResource;
use App\Models\Currency;
use App\Models\Notification;
use App\Models\PageSetting;
use App\Models\Plugin;
use App\Models\Setting;
use App\Models\UserNavigation;
use App\Models\WithdrawMethod;
use App\Traits\ApiResponseTrait;
use Illuminate\Http\Request;

class GeneralController extends Controller
{
    use ApiResponseTrait;

    public function getNotificationTypes($type)
    {
        $template = \App\Models\Template::where('for', $type)->pluck('code');

        return response()->json([
            'status' => true,
            'data' => $template,
        ]);
    }

    public function getCountries()
    {
        $location = getLocation();

        return response()->json([
            'status' => true,
            'data' => collect(getCountries())->map(function ($country) use ($location) {
                $country['selected'] = $country['code'] == $location->country_code;

                return $country;
            }),
        ]);
    }

    public function getCurrencies()
    {
        $multiCurrencyEnabled = setting('multiple_currency', 'permission');
        if (! $multiCurrencyEnabled) {
            return response()->json([
                'status' => false,
                'message' => 'Multiple currency is disabled',
                'data' => [],
            ]);
        }

        $currencies = Currency::all();

        $currencies->transform(function ($currency) {
            $currency->full_name = $currency->name . ' (' . $currency->code . ')';

            return $currency;
        });

        return response()->json([
            'status' => true,
            'data' => $currencies->toArray(),
        ]);
    }

    public function getSettings(Request $request)
    {
        $type = $request->get('key', 'all');
        $settings = Setting::select('name', 'val')->get()->map(function ($setting) {
            return [
                'name' => $setting->name,
                'value' => file_exists(base_path('assets/' . $setting->val)) ? asset($setting->val) : $setting->val,
            ];
        });

        return response()->json([
            'status' => true,
            'data' => $type == 'all' ? $settings : data_get(collect($settings)->firstWhere('name', $type), 'value'),
        ]);
    }

    public function getLanguages()
    {
        if (! setting('language_switcher')) {
            return response()->json([
                'status' => false,
                'message' => 'Language switcher is disabled',
                'data' => [],
            ]);
        }
        $languages = \App\Models\Language::where('status', 1)->get();

        return response()->json([
            'status' => true,
            'data' => $languages->toArray(),
        ]);
    }

    public function getRegisterFields($type)
    {
        $registerFields = PageSetting::select('key', 'value')->when($type == 'merchant' || $type == 'agent', function ($query) use ($type) {
            $query->whereLike('key', $type . '_%');
        })->when($type == 'user', function ($query) {
            $query->whereNotLike('key', 'merchant_%')->whereNotLike('key', 'agent_%');
        })->whereNotIn('key', ['shape_one', 'shape_two', 'shape_three', 'basic_page_background', 'breadcrumb'])->get();

        return response()->json([
            'status' => true,
            'data' => $registerFields,
        ]);
    }

    public function getKycStatus()
    {
        return response()->json([
            'status' => true,
            'data' => [
                'status' => [
                    'not_submitted' => KYCStatus::NOT_SUBMITTED->value,
                    'pending' => KYCStatus::Pending->value,
                    'verified' => KYCStatus::Verified->value,
                    'failed' => KYCStatus::Failed->value,
                ],
            ],
        ]);
    }

    public function getTransactionTypesAndStatuses()
    {
        $transactionTypes = collect(TxnType::cases())->map(function ($txnType) {
            return [
                'name' => str($txnType->value)->headline(),
                'value' => $txnType->value,
            ];
        });

        // User transaction types (based on isPlusTransaction helper)
        $userTransactionTypes = collect([
            TxnType::Deposit,
            TxnType::ManualDeposit,
            TxnType::Withdraw,
            TxnType::WithdrawAuto,
            TxnType::SendMoney,
            TxnType::ReceiveMoney,
            TxnType::CashReceived,
            TxnType::Refund,
            TxnType::Referral,
            TxnType::Exchange,
            TxnType::GiftRedeemed,
            TxnType::SignupBonus,
            TxnType::CashIn,
            TxnType::CashOut,
            TxnType::Credit,
            TxnType::Debit,
            TxnType::RequestMoney,
            TxnType::Invoice,
        ])->map(function ($txnType) {
            return [
                'name' => str($txnType->value)->headline(),
                'value' => $txnType->value,
            ];
        });

        // Agent transaction types (based on isAgentPlusTransaction and agent-specific transactions)
        $agentTransactionTypes = collect([
            TxnType::Deposit,
            TxnType::ManualDeposit,
            TxnType::CashoutCommission,
            TxnType::CashInCommission,
            TxnType::Credit,
            TxnType::Debit,
            TxnType::Refund,
            TxnType::SignupBonus,
            TxnType::CashReceived,
            TxnType::CashIn,
            TxnType::CashOut,
            TxnType::Withdraw,
            TxnType::WithdrawAuto,
        ])->map(function ($txnType) {
            return [
                'name' => str($txnType->value)->headline(),
                'value' => $txnType->value,
            ];
        });

        // Merchant transaction types (based on isMerchantPlusTransaction and merchant-specific transactions)
        $merchantTransactionTypes = collect([
            TxnType::Payment,
            TxnType::Credit,
            TxnType::Debit,
            TxnType::Refund,
            TxnType::SignupBonus,
            TxnType::Withdraw,
            TxnType::WithdrawAuto,
            TxnType::Deposit,
            TxnType::ManualDeposit,
        ])->map(function ($txnType) {
            return [
                'name' => str($txnType->value)->headline(),
                'value' => $txnType->value,
            ];
        });

        return response()->json([
            'status' => true,
            'data' => [
                'user_types' => $userTransactionTypes,
                'agent_types' => $agentTransactionTypes,
                'merchant_types' => $merchantTransactionTypes,
                'statuses' => collect(TxnStatus::cases())->map(function ($txnStatus) {
                    return [
                        'name' => str($txnStatus->value)->headline(),
                        'value' => $txnStatus->value,
                    ];
                }),
            ],
        ]);
    }

    public function getWithdrawMethods()
    {
        $methods = WithdrawMethod::where('status', 1)->get()->map(function ($method) {
            $method->fields = json_encode((object) json_decode($method->fields, true));

            return $method;
        });

        return response()->json([
            'status' => true,
            'data' => $methods,
        ]);
    }

    public function getOnboardingScreenImages()
    {
        return response()->json([
            'status' => true,
            'data' => [
                asset(getPageSetting('app_splash_one_image')),
                asset(getPageSetting('app_splash_two_image')),
                asset(getPageSetting('app_splash_three_image')),
                asset(getPageSetting('app_splash_four_image')),
            ],
        ]);
    }

    public function getNavigation()
    {
        $user_navigations = UserNavigation::orderBy('position')->get();

        return response()->json([
            'status' => true,
            'data' => NavigationResource::collection($user_navigations),
        ]);
    }

    public function getPlugins()
    {
        $plugins = Plugin::where('status', 1)->get();

        return response()->json([
            'status' => true,
            'data' => $plugins,
        ]);
    }

    public function getNotifications(Request $request)
    {
        $query = Notification::whereNot('for', 'admin')->where('user_id', auth()->id());

        $notifications = $query->clone()->latest()->paginate($request->integer('per_page', 15));

        $data = [
            'notifications' => NotificationResource::collection($notifications),
            'unread_count' => $query->clone()->where('read', 0)->count(),
            'meta' => [
                'current_page' => $notifications->currentPage(),
                'last_page' => $notifications->lastPage(),
                'per_page' => $notifications->perPage(),
                'total' => $notifications->total(),
            ],
        ];

        return $this->success($data, 'Notifications');
    }

    public function convertCurrency($amount, $currencyCode, $thousandSeparatorRemove = 'true', $fromCurrency = null)
    {
        $fromCurrency = $fromCurrency ?? setting('site_currency');
        $converted = currency()->convert($amount, $fromCurrency, $currencyCode);
        $converted = $thousandSeparatorRemove == 'true' ? currency()->thousandSeparatorRemove($converted) : $converted;

        return $this->success([
            'base_currency' => $fromCurrency,
            'target_currency' => $currencyCode,
            'base_amount' => $amount,
            'converted_amount' => $converted,
            'rate' => (string) (currency()->getCurrencyRate($currencyCode) / currency()->getCurrencyRate($fromCurrency)),
            'thousandSeparatorRemove' => $thousandSeparatorRemove == 'true',
        ]);
    }

    public function markNotification(Request $request)
    {
        $request->validate([
            'for' => 'required|in:user,merchant,agent',
        ]);

        $user = $request->user();

        Notification::where('for', $request->get('for'))->where('user_id', $user->id)->update(['read' => true]);

        return $this->success(null, __('All Notifications marked as read'));
    }
}
