<?php

namespace App\Http\Controllers\Api\Agent;

use App\Http\Controllers\Controller;
use App\Models\Notification;
use App\Models\UserDevice;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class NotificationController extends Controller
{
    public function registerDevice(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'device_id' => 'required|string',
            'device_type' => 'required|in:android,ios',
            'fcm_token' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => false,
                'message' => $validator->errors()->first(),
            ], 422);
        }

        $user = $request->user();

        // Update or create device
        UserDevice::updateOrCreate(
            [
                'user_id' => $user->id,
            ],
            [
                'device_id' => $request->device_id,
                'device_type' => $request->device_type,
                'fcm_token' => $request->fcm_token,
            ]
        );

        return response()->json([
            'status' => true,
            'message' => 'Device registered successfully',
        ]);
    }

    public function getNotifications(Request $request)
    {
        $user = auth()->user();
        $page = $request->page ?? 1;
        $limit = $request->limit ?? 15;

        $notifications = Notification::where('for', 'user')
            ->where('user_id', $user->id)
            ->latest()
            ->paginate($limit, ['*'], 'page', $page);

        return response()->json([
            'status' => true,
            'data' => $notifications,
            'pagination' => [
                'current_page' => $notifications->currentPage(),
                'last_page' => $notifications->lastPage(),
                'per_page' => $notifications->perPage(),
                'total' => $notifications->total(),
            ],
        ]);
    }

    public function markAsRead(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'notification_id' => 'required|exists:notifications,id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => false,
                'message' => $validator->errors()->first(),
            ], 422);
        }

        $user = auth()->user();
        $notification = Notification::where('id', $request->notification_id)
            ->where('user_id', $user->id)
            ->first();

        if (! $notification) {
            return response()->json([
                'status' => false,
                'message' => 'Notification not found',
            ], 404);
        }

        $notification->update(['read' => 1]);

        return response()->json([
            'status' => true,
            'message' => 'Notification marked as read',
        ]);
    }
}
